<?php
/**
 * Alpi Diş Hastaneleri CRM - Follow-up Status Scheduler
 * Follow-up durumlarını otomatik günceller (cron job için)
 * 
 * Bu script günde 1 kez çalıştırılmalıdır:
 * 0 9 * * * /usr/bin/php /path/to/your/project/scheduled/update_followup_status.php
 */

// Set script path for includes
$script_dir = dirname(__FILE__);
require_once $script_dir . '/../config/config.php';
require_once $script_dir . '/../config/database.php';
require_once $script_dir . '/../config/auth.php';

try {
    $db = getDB();
    
    echo "Follow-up Status Scheduler başlatıldı: " . date('Y-m-d H:i:s') . "\n";
    
    // 1. Gecikmiş follow-up'ları işaretle (pending -> overdue)
    $sql = "UPDATE followups 
            SET status = 'overdue', updated_at = NOW() 
            WHERE status = 'pending' 
            AND followup_date < CURDATE()";
    
    $stmt = $db->prepare($sql);
    $stmt->execute();
    $overdue_count = $stmt->rowCount();
    
    echo "Gecikmiş follow-up sayısı: $overdue_count\n";
    
    // 2. Acil follow-up bildirimlerini oluştur
    $sql = "SELECT f.id, f.patient_id, f.followup_date, 
                   p.first_name, p.last_name, p.phone, p.branch_id
            FROM followups f
            INNER JOIN patients p ON f.patient_id = p.id
            WHERE f.status = 'pending' 
            AND f.followup_date < DATE_SUB(CURDATE(), INTERVAL 3 DAY)
            AND NOT EXISTS (
                SELECT 1 FROM followup_notifications fn 
                WHERE fn.followup_id = f.id AND fn.notification_type = 'urgent'
            )";
    
    $stmt = $db->prepare($sql);
    $stmt->execute();
    $urgent_followups = $stmt->fetchAll();
    
    $urgent_count = 0;
    foreach ($urgent_followups as $followup) {
        // Acil bildirim oluştur
        $stmt = $db->prepare("INSERT INTO followup_notifications 
                            (followup_id, notification_type, created_at) 
                            VALUES (?, 'urgent', NOW())");
        $stmt->execute([$followup['id']]);
        $urgent_count++;
        
        echo "Acil bildirim oluşturuldu: {$followup['first_name']} {$followup['last_name']} - Follow-up Tarihi: {$followup['followup_date']}\n";
    }
    
    echo "Acil bildirim sayısı: $urgent_count\n";
    
    // 3. Maksimum deneme sayısına ulaşan follow-up'ları iptal et
    $sql = "SELECT f.patient_id, COUNT(*) as attempt_count,
                   p.first_name, p.last_name
            FROM followups f
            INNER JOIN patients p ON f.patient_id = p.id
            WHERE f.result IN ('no_answer', 'busy', 'wrong_number')
            AND f.status = 'completed'
            GROUP BY f.patient_id
            HAVING attempt_count >= 3
            AND EXISTS (
                SELECT 1 FROM followups f2 
                WHERE f2.patient_id = f.patient_id 
                AND f2.status = 'pending'
            )";
    
    $stmt = $db->prepare($sql);
    $stmt->execute();
    $max_attempt_patients = $stmt->fetchAll();
    
    $cancelled_count = 0;
    foreach ($max_attempt_patients as $patient) {
        // Bekleyen follow-up'ları iptal et
        $stmt = $db->prepare("UPDATE followups 
                            SET status = 'cancelled', updated_at = NOW() 
                            WHERE patient_id = ? AND status = 'pending'");
        $stmt->execute([$patient['patient_id']]);
        
        // Hasta durumunu kaybedildi olarak işaretle
        $stmt = $db->prepare("UPDATE patients 
                            SET status = 'lost', updated_at = NOW() 
                            WHERE id = ?");
        $stmt->execute([$patient['patient_id']]);
        
        $cancelled_count++;
        echo "Maksimum deneme aşıldı: {$patient['first_name']} {$patient['last_name']} - Follow-up iptal edildi\n";
    }
    
    echo "İptal edilen follow-up sayısı: $cancelled_count\n";
    
    // 4. İstatistikleri güncelle (sistem performansı için cache)
    $stats_sql = "INSERT INTO settings (setting_key, setting_value, description, updated_at) VALUES
                  ('daily_stats_date', CURDATE(), 'Son istatistik güncelleme tarihi', NOW()),
                  ('pending_followups', 
                   (SELECT COUNT(*) FROM followups WHERE status = 'pending'), 
                   'Bekleyen follow-up sayısı', NOW()),
                  ('overdue_followups', 
                   (SELECT COUNT(*) FROM followups WHERE status = 'overdue'), 
                   'Gecikmiş follow-up sayısı', NOW()),
                  ('urgent_followups', 
                   (SELECT COUNT(*) FROM followup_notifications WHERE notification_type = 'urgent' AND acknowledged_at IS NULL), 
                   'Acil follow-up sayısı', NOW())
                  ON DUPLICATE KEY UPDATE 
                  setting_value = VALUES(setting_value),
                  updated_at = VALUES(updated_at)";
    
    $stmt = $db->prepare($stats_sql);
    $stmt->execute();
    
    // 5. Eski bildirimları temizle (30 gün öncesi)
    $stmt = $db->prepare("DELETE FROM followup_notifications 
                        WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
    $stmt->execute();
    $cleaned_notifications = $stmt->rowCount();
    
    echo "Temizlenen eski bildirim sayısı: $cleaned_notifications\n";
    
    // 6. Aktivite loglarını temizle (90 gün öncesi)
    $stmt = $db->prepare("DELETE FROM activity_log 
                        WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)");
    $stmt->execute();
    $cleaned_logs = $stmt->rowCount();
    
    echo "Temizlenen eski log sayısı: $cleaned_logs\n";
    
    echo "Follow-up Status Scheduler tamamlandı: " . date('Y-m-d H:i:s') . "\n";
    echo "Özet: $overdue_count gecikmiş, $urgent_count acil, $cancelled_count iptal edildi\n";
    echo "----------------------------------------\n";
    
} catch (Exception $e) {
    echo "HATA: " . $e->getMessage() . "\n";
    echo "Hata Zamanı: " . date('Y-m-d H:i:s') . "\n";
    echo "----------------------------------------\n";
    
    // Log the error to activity_log if possible
    try {
        if (isset($db)) {
            $stmt = $db->prepare("INSERT INTO activity_log 
                                (user_id, action_type, target_table, description, created_at) 
                                VALUES (NULL, 'system_error', 'scheduler', ?, NOW())");
            $stmt->execute(["Follow-up Scheduler Hatası: " . $e->getMessage()]);
        }
    } catch (Exception $log_error) {
        echo "Log yazılamadı: " . $log_error->getMessage() . "\n";
    }
    
    exit(1);
}

exit(0);
?>