/**
 * Alpi Diş Hastaneleri CRM - Main JavaScript
 * Genel JavaScript fonksiyonları ve yardımcılar
 */

// App namespace
window.AlpiCRM = window.AlpiCRM || {};

// Main app object
AlpiCRM = {
    // Configuration
    config: {
        ajaxTimeout: 30000,
        dateFormat: 'dd.mm.yyyy',
        timeFormat: 'HH:mm'
    },

    // Initialize app
    init: function() {
        this.bindEvents();
        this.initComponents();
        this.setupCSRF();
    },

    // Bind global events
    bindEvents: function() {
        // Form validation
        $(document).on('submit', 'form[data-validate]', this.validateForm);
        
        // Confirm dialogs
        $(document).on('click', '[data-confirm]', this.confirmAction);
        
        // Auto phone formatting
        $(document).on('input', 'input[type="tel"], input[name*="phone"]', this.formatPhone);
        
        // Search functionality
        $(document).on('input', '[data-search]', this.debounce(this.handleSearch, 300));
        
        // Ajax forms
        $(document).on('submit', 'form[data-ajax]', this.handleAjaxForm);
        
        // Tooltip initialization
        $('[data-bs-toggle="tooltip"]').tooltip();
        
        // Popover initialization  
        $('[data-bs-toggle="popover"]').popover();
    },

    // Initialize UI components
    initComponents: function() {
        // Initialize date pickers
        this.initDatePickers();
        
        // Initialize select2 dropdowns if available
        if (typeof $.fn.select2 !== 'undefined') {
            $('.select2').select2({
                theme: 'bootstrap-5',
                width: '100%'
            });
        }
        
        // Auto-hide alerts after 5 seconds
        $('.alert:not(.alert-permanent)').delay(5000).fadeOut();
        
        // Initialize charts if available
        this.initCharts();
    },

    // Setup CSRF token for all AJAX requests
    setupCSRF: function() {
        const token = $('meta[name="csrf-token"]').attr('content');
        if (token) {
            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': token
                }
            });
        }
    },

    // Initialize date pickers
    initDatePickers: function() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.datepicker', {
                dateFormat: 'd.m.Y',
                locale: 'tr'
            });
            
            flatpickr('.datetimepicker', {
                enableTime: true,
                dateFormat: 'd.m.Y H:i',
                locale: 'tr'
            });
        }
    },

    // Phone number formatting
    formatPhone: function() {
        let value = $(this).val().replace(/\D/g, '');
        
        if (value.length > 0) {
            if (value.length <= 11) {
                if (value.startsWith('0')) {
                    // Format: 0XXX XXX XX XX
                    value = value.replace(/(\d{4})(\d{3})(\d{2})(\d{2})/, '$1 $2 $3 $4');
                } else if (value.startsWith('5') && value.length <= 10) {
                    // Format: 5XX XXX XX XX
                    value = value.replace(/(\d{3})(\d{3})(\d{2})(\d{2})/, '$1 $2 $3 $4');
                }
            }
        }
        
        $(this).val(value);
    },

    // Form validation
    validateForm: function(e) {
        const form = $(this);
        let isValid = true;

        // Clear previous errors
        form.find('.is-invalid').removeClass('is-invalid');
        form.find('.invalid-feedback').remove();

        // Required fields
        form.find('[required]').each(function() {
            const field = $(this);
            const value = field.val().trim();

            if (!value) {
                AlpiCRM.showFieldError(field, 'Bu alan zorunludur.');
                isValid = false;
            }
        });

        // Email validation
        form.find('input[type="email"]').each(function() {
            const field = $(this);
            const value = field.val().trim();
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

            if (value && !emailRegex.test(value)) {
                AlpiCRM.showFieldError(field, 'Geçerli bir e-posta adresi giriniz.');
                isValid = false;
            }
        });

        // Phone validation
        form.find('input[type="tel"], input[name*="phone"]').each(function() {
            const field = $(this);
            const value = field.val().replace(/\D/g, '');

            if (value && !AlpiCRM.isValidPhone(value)) {
                AlpiCRM.showFieldError(field, 'Geçerli bir telefon numarası giriniz.');
                isValid = false;
            }
        });

        if (!isValid) {
            e.preventDefault();
            return false;
        }
    },

    // Show field error
    showFieldError: function(field, message) {
        field.addClass('is-invalid');
        const feedback = $('<div class="invalid-feedback">' + message + '</div>');
        if (field.parent().hasClass('input-group')) {
            field.parent().after(feedback);
        } else {
            field.after(feedback);
        }
    },

    // Phone validation
    isValidPhone: function(phone) {
        phone = phone.replace(/\D/g, '');
        return /^(5[0-9]{9}|0?5[0-9]{9})$/.test(phone);
    },

    // Confirm action dialog
    confirmAction: function(e) {
        const message = $(this).data('confirm') || 'Bu işlemi gerçekleştirmek istediğinizden emin misiniz?';
        
        if (!confirm(message)) {
            e.preventDefault();
            return false;
        }
    },

    // Handle search with debounce
    handleSearch: function() {
        const input = $(this);
        const target = input.data('search');
        const query = input.val().trim();

        if (query.length >= 2) {
            AlpiCRM.performSearch(target, query);
        } else if (query.length === 0) {
            AlpiCRM.clearSearch(target);
        }
    },

    // Perform search
    performSearch: function(target, query) {
        // Implementation depends on the specific search target
        console.log('Searching:', target, query);
    },

    // Clear search
    clearSearch: function(target) {
        // Implementation depends on the specific search target
        console.log('Clear search:', target);
    },

    // Handle AJAX forms
    handleAjaxForm: function(e) {
        e.preventDefault();
        
        const form = $(this);
        const submitBtn = form.find('[type="submit"]');
        const originalText = submitBtn.text();
        
        // Disable submit button and show loading
        submitBtn.prop('disabled', true)
                 .html('<span class="spinner-border spinner-border-sm me-2"></span>Gönderiliyor...');
        
        $.ajax({
            url: form.attr('action') || window.location.href,
            method: form.attr('method') || 'POST',
            data: form.serialize(),
            timeout: AlpiCRM.config.ajaxTimeout,
            success: function(response) {
                if (response.success) {
                    AlpiCRM.showAlert(response.message || 'İşlem başarıyla tamamlandı.', 'success');
                    
                    if (response.redirect) {
                        setTimeout(() => window.location.href = response.redirect, 1000);
                    } else if (response.reload) {
                        setTimeout(() => window.location.reload(), 1000);
                    }
                } else {
                    AlpiCRM.showAlert(response.message || 'Bir hata oluştu.', 'danger');
                }
            },
            error: function(xhr, status, error) {
                let message = 'Bir hata oluştu. Lütfen tekrar deneyin.';
                
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                } else if (status === 'timeout') {
                    message = 'İşlem zaman aşımına uğradı. Lütfen tekrar deneyin.';
                }
                
                AlpiCRM.showAlert(message, 'danger');
            },
            complete: function() {
                // Re-enable submit button
                submitBtn.prop('disabled', false).text(originalText);
            }
        });
    },

    // Show alert message
    showAlert: function(message, type = 'info', duration = 5000) {
        const alertHtml = `
            <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `;
        
        // Remove existing alerts
        $('.alert:not(.alert-permanent)').remove();
        
        // Add new alert
        $('main').prepend(alertHtml);
        
        // Auto-hide after duration
        if (duration > 0) {
            setTimeout(() => $('.alert:not(.alert-permanent)').fadeOut(), duration);
        }
    },

    // Debounce function
    debounce: function(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func.apply(this, args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    },

    // Initialize charts
    initCharts: function() {
        // Chart.js default configuration
        if (typeof Chart !== 'undefined') {
            Chart.defaults.font.family = 'Inter';
            Chart.defaults.font.size = 12;
            Chart.defaults.color = '#6b7280';
            Chart.defaults.plugins.legend.position = 'bottom';
        }
    },

    // Utility functions
    utils: {
        // Format date
        formatDate: function(date, format = 'dd.mm.yyyy') {
            if (!date) return '';
            
            const d = new Date(date);
            const day = String(d.getDate()).padStart(2, '0');
            const month = String(d.getMonth() + 1).padStart(2, '0');
            const year = d.getFullYear();
            
            return format.replace('dd', day)
                        .replace('mm', month)
                        .replace('yyyy', year);
        },

        // Format currency
        formatCurrency: function(amount, currency = 'TL') {
            return new Intl.NumberFormat('tr-TR', {
                style: 'currency',
                currency: 'TRY',
                minimumFractionDigits: 2
            }).format(amount);
        },

        // Format phone number
        formatPhoneDisplay: function(phone) {
            if (!phone) return '';
            
            const cleaned = phone.replace(/\D/g, '');
            if (cleaned.length === 11 && cleaned.startsWith('0')) {
                return cleaned.replace(/(\d{4})(\d{3})(\d{2})(\d{2})/, '$1 $2 $3 $4');
            }
            return phone;
        },

        // Get query parameter
        getUrlParameter: function(name) {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(name);
        },

        // Copy to clipboard
        copyToClipboard: function(text) {
            if (navigator.clipboard) {
                navigator.clipboard.writeText(text).then(() => {
                    AlpiCRM.showAlert('Panoya kopyalandı!', 'success', 2000);
                });
            } else {
                // Fallback for older browsers
                const textArea = document.createElement('textarea');
                textArea.value = text;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                AlpiCRM.showAlert('Panoya kopyalandı!', 'success', 2000);
            }
        }
    }
};

// Initialize when document is ready
$(document).ready(function() {
    AlpiCRM.init();
});

// Export for use in other scripts
window.AlpiCRM = AlpiCRM;