<?php
/**
 * Alpi Diş Hastaneleri CRM - User Profile
 * Kullanıcı profil yönetimi
 */

require_once __DIR__ . '/../../includes/header.php';

$page_title = 'Profil';
$breadcrumb = [
    ['text' => 'Profil']
];

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

try {
    $db = getDB();
    
    // Get user data
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        throw new Exception('Kullanıcı bulunamadı.');
    }
    
    // Handle form submissions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        
        // Validate CSRF token
        if (!Auth::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Güvenlik doğrulaması başarısız.');
        }
        
        if (isset($_POST['update_profile'])) {
            // Update profile information
            $full_name = sanitizeInput($_POST['full_name']);
            $email = sanitizeInput($_POST['email']);
            $phone = sanitizeInput($_POST['phone']);
            
            // Validate inputs
            if (empty($full_name) || empty($email)) {
                throw new Exception('Ad soyad ve e-posta zorunludur.');
            }
            
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                throw new Exception('Geçerli bir e-posta adresi giriniz.');
            }
            
            // Check if email is already used by another user
            $stmt = $db->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
            $stmt->execute([$email, $user_id]);
            if ($stmt->fetch()) {
                throw new Exception('Bu e-posta adresi başka bir kullanıcı tarafından kullanılmaktadır.');
            }
            
            // Update user
            $stmt = $db->prepare("UPDATE users SET full_name = ?, email = ?, phone = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$full_name, $email, $phone, $user_id]);
            
            // Update session
            $_SESSION['full_name'] = $full_name;
            
            // Log activity
            Auth::logActivity($user_id, 'update', 'users', $user_id, 'Profil bilgileri güncellendi');
            
            $success_message = 'Profil bilgileriniz başarıyla güncellendi.';
            
            // Refresh user data
            $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch();
            
        } elseif (isset($_POST['change_password'])) {
            // Change password
            $current_password = $_POST['current_password'];
            $new_password = $_POST['new_password'];
            $confirm_password = $_POST['confirm_password'];
            
            // Validate inputs
            if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                throw new Exception('Tüm şifre alanları zorunludur.');
            }
            
            if (!password_verify($current_password, $user['password_hash'])) {
                throw new Exception('Mevcut şifreniz yanlış.');
            }
            
            if (strlen($new_password) < PASSWORD_MIN_LENGTH) {
                throw new Exception('Yeni şifre en az ' . PASSWORD_MIN_LENGTH . ' karakter olmalıdır.');
            }
            
            if ($new_password !== $confirm_password) {
                throw new Exception('Yeni şifreler eşleşmiyor.');
            }
            
            // Update password
            $password_hash = Auth::hashPassword($new_password);
            $stmt = $db->prepare("UPDATE users SET password_hash = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$password_hash, $user_id]);
            
            // Log activity
            Auth::logActivity($user_id, 'update', 'users', $user_id, 'Şifre değiştirildi');
            
            $success_message = 'Şifreniz başarıyla değiştirildi.';
            
        } elseif (isset($_POST['upload_avatar'])) {
            // Handle avatar upload
            if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
                
                $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                $max_size = 2 * 1024 * 1024; // 2MB
                
                $file = $_FILES['avatar'];
                
                // Validate file type
                if (!in_array($file['type'], $allowed_types)) {
                    throw new Exception('Sadece JPG, PNG ve GIF dosyaları yüklenebilir.');
                }
                
                // Validate file size
                if ($file['size'] > $max_size) {
                    throw new Exception('Dosya boyutu 2MB\'dan büyük olamaz.');
                }
                
                // Create upload directory if not exists
                $upload_dir = __DIR__ . '/../../uploads/avatars/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                // Generate unique filename
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = 'avatar_' . $user_id . '_' . time() . '.' . $extension;
                $filepath = $upload_dir . $filename;
                
                // Delete old avatar if exists
                if ($user['avatar'] && file_exists($upload_dir . $user['avatar'])) {
                    unlink($upload_dir . $user['avatar']);
                }
                
                // Move uploaded file
                if (move_uploaded_file($file['tmp_name'], $filepath)) {
                    // Update database
                    $stmt = $db->prepare("UPDATE users SET avatar = ?, updated_at = NOW() WHERE id = ?");
                    $stmt->execute([$filename, $user_id]);
                    
                    // Log activity
                    Auth::logActivity($user_id, 'update', 'users', $user_id, 'Profil fotoğrafı güncellendi');
                    
                    $success_message = 'Profil fotoğrafınız başarıyla güncellendi.';
                    
                    // Refresh user data
                    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
                    $stmt->execute([$user_id]);
                    $user = $stmt->fetch();
                    
                } else {
                    throw new Exception('Dosya yüklenirken bir hata oluştu.');
                }
            } else {
                throw new Exception('Dosya seçiniz.');
            }
        }
    }
    
} catch (Exception $e) {
    $error_message = $e->getMessage();
}

// Role names
$role_names = [
    'owner' => 'Patron',
    'admin' => 'Yönetici',
    'sales' => 'Satış',
    'callcenter' => 'Çağrı Merkezi'
];
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Profil</h1>
</div>

<?php if ($success_message): ?>
<div class="alert alert-success" role="alert">
    <i class="fas fa-check-circle me-2"></i>
    <?php echo htmlspecialchars($success_message); ?>
</div>
<?php endif; ?>

<?php if ($error_message): ?>
<div class="alert alert-danger" role="alert">
    <i class="fas fa-exclamation-triangle me-2"></i>
    <?php echo htmlspecialchars($error_message); ?>
</div>
<?php endif; ?>

<div class="row">
    <!-- Profile Picture -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-camera me-2"></i>
                    Profil Fotoğrafı
                </h5>
            </div>
            <div class="card-body text-center">
                <?php if ($user['avatar']): ?>
                <img src="/uploads/avatars/<?php echo htmlspecialchars($user['avatar']); ?>" 
                     alt="Profil Fotoğrafı" class="rounded-circle mb-3" width="150" height="150" style="object-fit: cover;">
                <?php else: ?>
                <div class="bg-secondary rounded-circle d-inline-flex align-items-center justify-content-center mb-3" 
                     style="width: 150px; height: 150px;">
                    <i class="fas fa-user fa-4x text-white"></i>
                </div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
                    <div class="mb-3">
                        <input type="file" class="form-control" name="avatar" accept="image/*" required>
                        <div class="form-text">JPG, PNG veya GIF. Maksimum 2MB.</div>
                    </div>
                    <button type="submit" name="upload_avatar" class="btn btn-primary">
                        <i class="fas fa-upload me-2"></i>
                        Fotoğraf Yükle
                    </button>
                </form>
            </div>
        </div>
        
        <!-- User Info Card -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="card-title mb-0">Kullanıcı Bilgileri</h6>
            </div>
            <div class="card-body">
                <div class="mb-2">
                    <small class="text-muted">Kullanıcı Adı:</small><br>
                    <strong><?php echo htmlspecialchars($user['username']); ?></strong>
                </div>
                <div class="mb-2">
                    <small class="text-muted">Rol:</small><br>
                    <span class="badge bg-primary"><?php echo $role_names[$user['role']] ?? $user['role']; ?></span>
                </div>
                <?php if ($user['branch_id']): ?>
                <div class="mb-2">
                    <small class="text-muted">Şube:</small><br>
                    <strong><?php echo getBranchName($user['branch_id']); ?></strong>
                </div>
                <?php endif; ?>
                <div class="mb-2">
                    <small class="text-muted">Kayıt Tarihi:</small><br>
                    <?php echo formatDateTime($user['created_at']); ?>
                </div>
                <div class="mb-0">
                    <small class="text-muted">Son Giriş:</small><br>
                    <?php echo $user['last_login'] ? formatDateTime($user['last_login']) : 'İlk giriş'; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Profile Forms -->
    <div class="col-lg-8">
        <!-- Profile Information -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-user me-2"></i>
                    Profil Bilgileri
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="full_name" class="form-label">Ad Soyad *</label>
                            <input type="text" class="form-control" id="full_name" name="full_name" 
                                   value="<?php echo htmlspecialchars($user['full_name']); ?>" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="email" class="form-label">E-posta *</label>
                            <input type="email" class="form-control" id="email" name="email" 
                                   value="<?php echo htmlspecialchars($user['email']); ?>" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="phone" class="form-label">Telefon</label>
                        <input type="tel" class="form-control" id="phone" name="phone" 
                               value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>">
                    </div>
                    
                    <button type="submit" name="update_profile" class="btn btn-primary">
                        <i class="fas fa-save me-2"></i>
                        Bilgileri Güncelle
                    </button>
                </form>
            </div>
        </div>
        
        <!-- Change Password -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-lock me-2"></i>
                    Şifre Değiştir
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
                    
                    <div class="mb-3">
                        <label for="current_password" class="form-label">Mevcut Şifre *</label>
                        <input type="password" class="form-control" id="current_password" name="current_password" required>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="new_password" class="form-label">Yeni Şifre *</label>
                            <input type="password" class="form-control" id="new_password" name="new_password" 
                                   minlength="<?php echo PASSWORD_MIN_LENGTH; ?>" required>
                            <div class="form-text">En az <?php echo PASSWORD_MIN_LENGTH; ?> karakter olmalıdır.</div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="confirm_password" class="form-label">Yeni Şifre (Tekrar) *</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                   minlength="<?php echo PASSWORD_MIN_LENGTH; ?>" required>
                        </div>
                    </div>
                    
                    <button type="submit" name="change_password" class="btn btn-warning">
                        <i class="fas fa-key me-2"></i>
                        Şifre Değiştir
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Password confirmation validation
document.getElementById('confirm_password').addEventListener('input', function() {
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = this.value;
    
    if (newPassword !== confirmPassword) {
        this.setCustomValidity('Şifreler eşleşmiyor');
    } else {
        this.setCustomValidity('');
    }
});

// File upload preview
document.querySelector('input[name="avatar"]').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            // Could add preview functionality here
        };
        reader.readAsDataURL(file);
    }
});
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>