<?php
/**
 * Alpi Diş Hastaneleri CRM - Follow-up Update
 * Çağrı merkezi follow-up sonuç güncelleme
 */

require_once __DIR__ . '/../../includes/header.php';

// Check permissions - only call center, admin, or owner can update follow-ups
Auth::requireRole([Auth::ROLE_CALLCENTER, Auth::ROLE_ADMIN, Auth::ROLE_OWNER]);

$page_title = 'Follow-Up Güncelle';
$breadcrumb = [
    ['text' => 'Follow-Up', 'url' => '/modules/followups/list.php'],
    ['text' => 'Güncelle']
];

$success_message = '';
$error_message = '';
$followup = null;
$patient = null;

try {
    $db = getDB();
    $followup_id = (int)($_GET['id'] ?? 0);
    
    if (!$followup_id) {
        throw new Exception('Geçersiz follow-up ID.');
    }
    
    // Get patient data
    $sql = "SELECT p.*, b.name as branch_name, c.name as consultant_name, r.reason_text
            FROM patients p
            LEFT JOIN branches b ON p.branch_id = b.id
            LEFT JOIN consultants c ON p.consultant_id = c.id
            LEFT JOIN reasons r ON p.reason_id = r.id
            WHERE p.id = ? AND p.status = 'pending'";
    
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $sql .= " AND p.branch_id = ?";
        $stmt = $db->prepare($sql);
        $stmt->execute([$patient_id, $_SESSION['branch_id']]);
    } else {
        $stmt = $db->prepare($sql);
        $stmt->execute([$patient_id]);
    }
    
    $patient = $stmt->fetch();
    
    if (!$patient) {
        throw new Exception('Hasta bulunamadı veya follow-up için uygun değil.');
    }
    
    // Check if patient can be called (not exceeded max attempts)
    if ($patient['followup_count'] >= MAX_FOLLOWUP_ATTEMPTS) {
        throw new Exception('Bu hasta maksimum arama sayısına ulaştı.');
    }
    
    // Get previous follow-ups
    $stmt = $db->prepare("SELECT f.*, u.full_name as contacted_by_name 
                         FROM followups f
                         LEFT JOIN users u ON f.contacted_by = u.id
                         WHERE f.patient_id = ? 
                         ORDER BY f.created_at DESC");
    $stmt->execute([$patient_id]);
    $followups = $stmt->fetchAll();
    
    // Handle form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        
        // Validate CSRF token
        if (!Auth::validateCSRFToken($_POST['csrf_token'] ?? '')) {
            throw new Exception('Güvenlik doğrulaması başarısız.');
        }
        
        $result = sanitizeInput($_POST['result']);
        $notes = sanitizeInput($_POST['notes']);
        $contact_method = sanitizeInput($_POST['contact_method']);
        $next_followup_date = $_POST['next_followup_date'] ?? null;
        
        // Validate inputs
        if (empty($result)) {
            throw new Exception('Arama sonucu seçiniz.');
        }
        
        $valid_results = ['answered', 'no_answer', 'busy', 'wrong_number', 'recovered', 'rejected'];
        if (!in_array($result, $valid_results)) {
            throw new Exception('Geçersiz arama sonucu.');
        }
        
        // Start transaction
        $db->beginTransaction();
        
        try {
            // Insert follow-up record
            $stmt = $db->prepare("INSERT INTO followups 
                                (patient_id, followup_date, contacted_by, contact_method, result, notes, next_followup_date, created_at) 
                                VALUES (?, CURDATE(), ?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$patient_id, $_SESSION['user_id'], $contact_method, $result, $notes, $next_followup_date]);
            
            // Update patient based on result
            $new_status = 'pending';
            $new_followup_date = null;
            $followup_count = $patient['followup_count'] + 1;
            
            switch ($result) {
                case 'recovered':
                    $new_status = 'recovered';
                    break;
                    
                case 'rejected':
                    $new_status = 'lost';
                    break;
                    
                case 'answered':
                case 'no_answer':
                case 'busy':
                    if ($next_followup_date && $followup_count < MAX_FOLLOWUP_ATTEMPTS) {
                        $new_followup_date = $next_followup_date;
                    } elseif ($followup_count >= MAX_FOLLOWUP_ATTEMPTS) {
                        $new_status = 'lost'; // Max attempts reached
                    }
                    break;
                    
                case 'wrong_number':
                    $new_status = 'lost';
                    break;
            }
            
            // Update patient
            $stmt = $db->prepare("UPDATE patients 
                                SET status = ?, followup_date = ?, followup_count = ?, updated_by = ?, updated_at = NOW() 
                                WHERE id = ?");
            $stmt->execute([$new_status, $new_followup_date, $followup_count, $_SESSION['user_id'], $patient_id]);
            
            // Log activity
            $status_text = [
                'pending' => 'Bekleyen',
                'recovered' => 'Geri Kazanılan',
                'lost' => 'Kaybedilen'
            ];
            
            Auth::logActivity($_SESSION['user_id'], 'update', 'patients', $patient_id, 
                             "Follow-up güncellendi: {$patient['first_name']} {$patient['last_name']} - Durum: {$status_text[$new_status]}");
            
            $db->commit();
            
            $success_message = 'Follow-up başarıyla kaydedildi.';
            
            // Redirect based on result
            if (in_array($new_status, ['recovered', 'lost'])) {
                $success_message .= ' Hasta durumu güncellendi.';
                // Redirect to list after 2 seconds
                echo "<script>setTimeout(() => window.location.href = '/modules/followups/list.php', 2000);</script>";
            } else {
                // Refresh patient data
                $stmt = $db->prepare($sql);
                if (!Auth::hasRole(Auth::ROLE_OWNER)) {
                    $stmt->execute([$patient_id, $_SESSION['branch_id']]);
                } else {
                    $stmt->execute([$patient_id]);
                }
                $patient = $stmt->fetch();
            }
            
        } catch (Exception $e) {
            $db->rollback();
            throw $e;
        }
    }
    
} catch (Exception $e) {
    $error_message = $e->getMessage();
    if (strpos($error_message, 'bulunamadı') !== false) {
        redirectTo('/modules/followups/list.php');
    }
}

$page_title = 'Follow-Up Güncelle';
$breadcrumb = [
    ['text' => 'Follow-Up', 'url' => '/modules/followups/list.php'],
    ['text' => 'Güncelle']
];
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">Follow-Up Güncelle</h1>
    <div class="btn-toolbar mb-2 mb-md-0">
        <a href="/modules/followups/list.php" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>
            Follow-Up Listesi
        </a>
    </div>
</div>

<?php if ($success_message): ?>
<div class="alert alert-success" role="alert">
    <i class="fas fa-check-circle me-2"></i>
    <?php echo htmlspecialchars($success_message); ?>
</div>
<?php endif; ?>

<?php if ($error_message): ?>
<div class="alert alert-danger" role="alert">
    <i class="fas fa-exclamation-triangle me-2"></i>
    <?php echo htmlspecialchars($error_message); ?>
</div>
<?php endif; ?>

<?php if (isset($patient)): ?>
<div class="row">
    <!-- Patient Info -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-user me-2"></i>
                    Hasta Bilgileri
                </h5>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <h4><?php echo htmlspecialchars($patient['first_name'] . ' ' . $patient['last_name']); ?></h4>
                    <?php if ($patient['patient_type'] === 'existing'): ?>
                    <span class="badge bg-secondary">Mevcut Hasta</span>
                    <?php endif; ?>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Telefon:</strong></div>
                    <div class="col-sm-8">
                        <a href="tel:<?php echo htmlspecialchars($patient['phone']); ?>" class="text-decoration-none">
                            <?php echo htmlspecialchars($patient['phone']); ?>
                        </a>
                    </div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Şube:</strong></div>
                    <div class="col-sm-8"><?php echo htmlspecialchars($patient['branch_name']); ?></div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Danışman:</strong></div>
                    <div class="col-sm-8"><?php echo htmlspecialchars($patient['consultant_name'] ?? '-'); ?></div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Kaynak:</strong></div>
                    <div class="col-sm-8"><?php echo htmlspecialchars($patient['source'] ?? '-'); ?></div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Başvuru:</strong></div>
                    <div class="col-sm-8"><?php echo formatDate($patient['application_date']); ?></div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Follow-up:</strong></div>
                    <div class="col-sm-8">
                        <?php 
                        $days_diff = (strtotime($patient['followup_date']) - strtotime('today')) / (60 * 60 * 24);
                        $class = '';
                        if ($days_diff < 0) {
                            $class = 'text-danger fw-bold';
                        } elseif ($days_diff == 0) {
                            $class = 'text-warning fw-bold';
                        }
                        ?>
                        <span class="<?php echo $class; ?>">
                            <?php echo formatDate($patient['followup_date']); ?>
                        </span>
                    </div>
                </div>
                
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Arama Sayısı:</strong></div>
                    <div class="col-sm-8">
                        <span class="badge <?php echo $patient['followup_count'] >= MAX_FOLLOWUP_ATTEMPTS ? 'bg-danger' : 'bg-primary'; ?>">
                            <?php echo $patient['followup_count']; ?> / <?php echo MAX_FOLLOWUP_ATTEMPTS; ?>
                        </span>
                    </div>
                </div>
                
                <?php if ($patient['reason_text']): ?>
                <div class="row mb-2">
                    <div class="col-sm-4"><strong>Neden:</strong></div>
                    <div class="col-sm-8">
                        <span class="badge bg-light text-dark">
                            <?php echo htmlspecialchars($patient['reason_text']); ?>
                        </span>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if ($patient['notes']): ?>
                <div class="row mb-2">
                    <div class="col-12"><strong>Notlar:</strong></div>
                    <div class="col-12">
                        <div class="alert alert-light">
                            <?php echo nl2br(htmlspecialchars($patient['notes'])); ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Follow-up Form -->
    <div class="col-lg-8">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-phone me-2"></i>
                    Arama Sonucu
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?php echo Auth::generateCSRFToken(); ?>">
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="contact_method" class="form-label">İletişim Yöntemi</label>
                            <select class="form-select" id="contact_method" name="contact_method" required>
                                <option value="phone">Telefon</option>
                                <option value="whatsapp">WhatsApp</option>
                                <option value="sms">SMS</option>
                                <option value="email">E-posta</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="result" class="form-label">Arama Sonucu *</label>
                            <select class="form-select" id="result" name="result" required>
                                <option value="">Sonuç Seçiniz</option>
                                <option value="answered">Görüştü</option>
                                <option value="no_answer">Açmadı</option>
                                <option value="busy">Meşgul</option>
                                <option value="wrong_number">Yanlış Numara</option>
                                <option value="recovered">Geri Kazanıldı</option>
                                <option value="rejected">Kesin Red</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mb-3" id="next_followup_group" style="display: none;">
                        <label for="next_followup_date" class="form-label">Sonraki Arama Tarihi</label>
                        <input type="date" class="form-control" id="next_followup_date" name="next_followup_date" 
                               min="<?php echo date('Y-m-d', strtotime('+1 day')); ?>">
                    </div>
                    
                    <div class="mb-4">
                        <label for="notes" class="form-label">Arama Notları</label>
                        <textarea class="form-control" id="notes" name="notes" rows="4" 
                                  placeholder="Arama ile ilgili detayları yazınız..."></textarea>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>
                            Follow-up Kaydet
                        </button>
                        <a href="/modules/followups/list.php" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-2"></i>
                            İptal
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Previous Follow-ups -->
        <?php if (!empty($followups)): ?>
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-history me-2"></i>
                    Önceki Aramalar
                </h6>
            </div>
            <div class="card-body">
                <div class="timeline">
                    <?php foreach ($followups as $followup): ?>
                    <div class="timeline-item mb-3">
                        <div class="d-flex">
                            <div class="flex-shrink-0">
                                <div class="timeline-marker bg-primary"></div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div>
                                        <strong><?php echo htmlspecialchars($followup['contacted_by_name']); ?></strong>
                                        <span class="badge bg-info ms-2"><?php echo ucfirst($followup['contact_method']); ?></span>
                                    </div>
                                    <small class="text-muted"><?php echo formatDateTime($followup['created_at']); ?></small>
                                </div>
                                <div class="mt-1">
                                    <?php
                                    $result_texts = [
                                        'answered' => 'Görüştü',
                                        'no_answer' => 'Açmadı',
                                        'busy' => 'Meşgul',
                                        'wrong_number' => 'Yanlış Numara',
                                        'recovered' => 'Geri Kazanıldı',
                                        'rejected' => 'Kesin Red'
                                    ];
                                    ?>
                                    <span class="badge bg-secondary">
                                        <?php echo $result_texts[$followup['result']] ?? $followup['result']; ?>
                                    </span>
                                </div>
                                <?php if ($followup['notes']): ?>
                                <div class="mt-2">
                                    <small><?php echo nl2br(htmlspecialchars($followup['notes'])); ?></small>
                                </div>
                                <?php endif; ?>
                                <?php if ($followup['next_followup_date']): ?>
                                <div class="mt-1">
                                    <small class="text-muted">
                                        Sonraki arama: <?php echo formatDate($followup['next_followup_date']); ?>
                                    </small>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<style>
.timeline-marker {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    margin-top: 4px;
}
</style>

<script>
// Show/hide next followup date based on result
document.getElementById('result').addEventListener('change', function() {
    const result = this.value;
    const nextFollowupGroup = document.getElementById('next_followup_group');
    const nextFollowupInput = document.getElementById('next_followup_date');
    
    if (['answered', 'no_answer', 'busy'].includes(result)) {
        nextFollowupGroup.style.display = 'block';
        nextFollowupInput.required = true;
    } else {
        nextFollowupGroup.style.display = 'none';
        nextFollowupInput.required = false;
        nextFollowupInput.value = '';
    }
});

// Confirm before submitting final results
document.querySelector('form').addEventListener('submit', function(e) {
    const result = document.getElementById('result').value;
    
    if (result === 'recovered') {
        if (!confirm('Hasta geri kazanıldı olarak işaretlenecek. Emin misiniz?')) {
            e.preventDefault();
        }
    } else if (result === 'rejected' || result === 'wrong_number') {
        if (!confirm('Hasta kaybedildi olarak işaretlenecek. Emin misiniz?')) {
            e.preventDefault();
        }
    }
});
</script>

<?php endif; ?>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>