<?php
/**
 * Alpi Diş Hastaneleri CRM - Activity Log List
 * İşlem geçmişi listesi
 */

require_once __DIR__ . '/../../includes/header.php';

// Check permissions
Auth::requireRole(Auth::ROLE_ADMIN);

$page_title = 'İşlem Geçmişi';
$breadcrumb = [
    ['text' => 'İşlem Geçmişi']
];

// Default values
$user_filter = $_GET['user'] ?? '';
$action_filter = $_GET['action'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = RECORDS_PER_PAGE;
$offset = ($page - 1) * $per_page;

try {
    $db = getDB();
    
    // Build base query
    $where_conditions = [];
    $params = [];
    
    // Branch access control (admin sees only their branch)
    if (!Auth::hasRole(Auth::ROLE_OWNER)) {
        $where_conditions[] = "(al.user_id IN (SELECT id FROM users WHERE branch_id = ?) OR al.user_id IS NULL)";
        $params[] = $_SESSION['branch_id'];
    }
    
    // User filter
    if ($user_filter) {
        $where_conditions[] = "al.user_id = ?";
        $params[] = $user_filter;
    }
    
    // Action filter
    if ($action_filter) {
        $where_conditions[] = "al.action_type = ?";
        $params[] = $action_filter;
    }
    
    // Date filters
    if ($date_from) {
        $where_conditions[] = "DATE(al.created_at) >= ?";
        $params[] = $date_from;
    }
    
    if ($date_to) {
        $where_conditions[] = "DATE(al.created_at) <= ?";
        $params[] = $date_to;
    }
    
    $where_sql = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Get total count
    $count_sql = "SELECT COUNT(*) as total FROM activity_log al $where_sql";
    $stmt = $db->prepare($count_sql);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];
    $total_pages = ceil($total_records / $per_page);
    
    // Get activity logs
    $sql = "SELECT al.*, u.full_name as user_name, u.role as user_role
            FROM activity_log al
            LEFT JOIN users u ON al.user_id = u.id
            $where_sql
            ORDER BY al.created_at DESC
            LIMIT $per_page OFFSET $offset";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $logs = $stmt->fetchAll();
    
    // Get users for filter
    $users = [];
    if (Auth::hasRole(Auth::ROLE_OWNER)) {
        $stmt = $db->prepare("SELECT id, full_name, role FROM users WHERE is_active = 1 ORDER BY full_name");
        $stmt->execute();
        $users = $stmt->fetchAll();
    } else {
        $stmt = $db->prepare("SELECT id, full_name, role FROM users WHERE branch_id = ? AND is_active = 1 ORDER BY full_name");
        $stmt->execute([$_SESSION['branch_id']]);
        $users = $stmt->fetchAll();
    }
    
    // Get action types
    $stmt = $db->prepare("SELECT DISTINCT action_type FROM activity_log ORDER BY action_type");
    $stmt->execute();
    $action_types = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
} catch (Exception $e) {
    showAlert('Veriler yüklenirken bir hata oluştu: ' . $e->getMessage(), 'danger');
    $logs = [];
    $total_records = 0;
    $total_pages = 1;
    $users = [];
    $action_types = [];
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2">İşlem Geçmişi</h1>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label for="user" class="form-label">Kullanıcı</label>
                <select class="form-select" id="user" name="user">
                    <option value="">Tüm Kullanıcılar</option>
                    <?php foreach ($users as $user): ?>
                    <option value="<?php echo $user['id']; ?>" 
                            <?php echo $user_filter == $user['id'] ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($user['full_name']); ?>
                        (<?php echo ucfirst($user['role']); ?>)
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label for="action" class="form-label">İşlem</label>
                <select class="form-select" id="action" name="action">
                    <option value="">Tüm İşlemler</option>
                    <?php foreach ($action_types as $action): ?>
                    <option value="<?php echo $action; ?>" 
                            <?php echo $action_filter === $action ? 'selected' : ''; ?>>
                        <?php echo ucfirst($action); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label for="date_from" class="form-label">Başlangıç</label>
                <input type="date" class="form-control" id="date_from" name="date_from" 
                       value="<?php echo htmlspecialchars($date_from); ?>">
            </div>
            <div class="col-md-2">
                <label for="date_to" class="form-label">Bitiş</label>
                <input type="date" class="form-control" id="date_to" name="date_to" 
                       value="<?php echo htmlspecialchars($date_to); ?>">
            </div>
            <div class="col-md-3 d-flex align-items-end gap-2">
                <button type="submit" class="btn btn-outline-primary">
                    <i class="fas fa-filter me-2"></i>
                    Filtrele
                </button>
                <a href="/modules/activity_log/list.php" class="btn btn-outline-secondary">
                    <i class="fas fa-times me-2"></i>
                    Temizle
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Activity Log Table -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <?php echo number_format($total_records); ?> işlem kaydı bulundu
        </h5>
    </div>
    <div class="card-body p-0">
        <?php if (!empty($logs)): ?>
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Tarih/Saat</th>
                        <th>Kullanıcı</th>
                        <th>İşlem</th>
                        <th>Tablo</th>
                        <th>Açıklama</th>
                        <th>IP Adresi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($logs as $log): ?>
                    <tr>
                        <td>
                            <small class="text-muted">
                                <?php echo formatDateTime($log['created_at']); ?>
                            </small>
                        </td>
                        <td>
                            <?php if ($log['user_name']): ?>
                                <div>
                                    <strong><?php echo htmlspecialchars($log['user_name']); ?></strong>
                                </div>
                                <small class="text-muted">
                                    <?php 
                                    $roles = [
                                        'owner' => 'Patron',
                                        'admin' => 'Yönetici',
                                        'sales' => 'Satış',
                                        'callcenter' => 'Çağrı Merkezi'
                                    ];
                                    echo $roles[$log['user_role']] ?? $log['user_role'];
                                    ?>
                                </small>
                            <?php else: ?>
                                <span class="text-muted">Sistem</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php
                            $action_badges = [
                                'create' => 'bg-success',
                                'update' => 'bg-primary',
                                'delete' => 'bg-danger',
                                'login' => 'bg-info',
                                'logout' => 'bg-secondary'
                            ];
                            $badge_class = $action_badges[$log['action_type']] ?? 'bg-light text-dark';
                            ?>
                            <span class="badge <?php echo $badge_class; ?>">
                                <?php echo ucfirst($log['action_type']); ?>
                            </span>
                        </td>
                        <td>
                            <code><?php echo htmlspecialchars($log['target_table']); ?></code>
                            <?php if ($log['record_id']): ?>
                                <br><small class="text-muted">ID: <?php echo $log['record_id']; ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($log['description']): ?>
                                <?php echo htmlspecialchars($log['description']); ?>
                            <?php else: ?>
                                <span class="text-muted">-</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <small class="text-muted">
                                <?php echo htmlspecialchars($log['ip_address'] ?? '-'); ?>
                            </small>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="card-footer">
            <nav aria-label="Activity log sayfalama">
                <ul class="pagination pagination-sm justify-content-center mb-0">
                    <!-- Previous Page -->
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">
                            <i class="fas fa-chevron-left"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                    
                    <!-- Page Numbers -->
                    <?php
                    $start_page = max(1, $page - 2);
                    $end_page = min($total_pages, $page + 2);
                    
                    for ($i = $start_page; $i <= $end_page; $i++):
                    ?>
                    <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                            <?php echo $i; ?>
                        </a>
                    </li>
                    <?php endfor; ?>
                    
                    <!-- Next Page -->
                    <?php if ($page < $total_pages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
            
            <div class="text-center mt-2">
                <small class="text-muted">
                    Sayfa <?php echo $page; ?> / <?php echo $total_pages; ?> 
                    (Toplam <?php echo number_format($total_records); ?> kayıt)
                </small>
            </div>
        </div>
        <?php endif; ?>
        
        <?php else: ?>
        <div class="text-center py-5">
            <i class="fas fa-history fa-3x text-muted mb-3"></i>
            <h5 class="text-muted">İşlem kaydı bulunamadı</h5>
            <p class="text-muted">Seçilen kriterlere göre işlem geçmişi bulunmamaktadır.</p>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>