<?php
/**
 * Alpi Diş Hastaneleri CRM - Authentication & Security Configuration
 * RBAC (Role-Based Access Control) ve güvenlik ayarları
 */

// Session security settings
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1);
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_samesite', 'Lax');

class Auth {
    
    // User roles
    const ROLE_OWNER = 'owner';        // Patron - Tüm erişim
    const ROLE_ADMIN = 'admin';        // Yönetici - Şube bazlı erişim
    const ROLE_SALES = 'sales';        // Satış - Hasta ekleme/düzenleme
    const ROLE_CALLCENTER = 'callcenter'; // Çağrı merkezi - Follow-up

    // Session timeout (30 minutes)
    const SESSION_TIMEOUT = 1800;
    
    // Failed login attempt limit
    const MAX_LOGIN_ATTEMPTS = 5;
    const LOCKOUT_TIME = 900; // 15 minutes

    public static function startSecureSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Session hijacking protection
        if (!isset($_SESSION['initiated'])) {
            session_regenerate_id(true);
            $_SESSION['initiated'] = true;
        }
        
        // Check session timeout
        if (isset($_SESSION['last_activity']) && 
            (time() - $_SESSION['last_activity'] > self::SESSION_TIMEOUT)) {
            self::logout();
            return false;
        }
        
        $_SESSION['last_activity'] = time();
        return true;
    }

    public static function login($username, $password) {
        try {
            $db = getDB();
            
            // Check failed attempts
            $ip = self::getUserIP();
            $stmt = $db->prepare("SELECT failed_attempts, last_attempt FROM login_attempts 
                                WHERE ip_address = ? AND last_attempt > DATE_SUB(NOW(), INTERVAL ? SECOND)");
            $stmt->execute([$ip, self::LOCKOUT_TIME]);
            $attempt = $stmt->fetch();
            
            if ($attempt && $attempt['failed_attempts'] >= self::MAX_LOGIN_ATTEMPTS) {
                throw new Exception("Çok fazla başarısız deneme. Lütfen " . (self::LOCKOUT_TIME/60) . " dakika bekleyin.");
            }
            
            // Get user
            $stmt = $db->prepare("SELECT id, username, password_hash, role, branch_id, is_active, 
                                full_name, email FROM users WHERE username = ? AND is_active = 1");
            $stmt->execute([$username]);
            $user = $stmt->fetch();
            
            if (!$user || !password_verify($password, $user['password_hash'])) {
                // Log failed attempt
                self::logFailedAttempt($ip);
                throw new Exception("Kullanıcı adı veya şifre hatalı.");
            }
            
            // Clear failed attempts
            self::clearFailedAttempts($ip);
            
            // Set session
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['branch_id'] = $user['branch_id'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['user_ip'] = $ip;
            $_SESSION['initiated'] = true;
            $_SESSION['last_activity'] = time();
            
            // Log successful login
            self::logActivity($user['id'], 'login', 'users', $user['id'], 'Kullanıcı giriş yaptı');
            
            return true;
            
        } catch (Exception $e) {
            throw $e;
        }
    }

    public static function logout() {
        if (isset($_SESSION['user_id'])) {
            self::logActivity($_SESSION['user_id'], 'logout', 'users', $_SESSION['user_id'], 'Kullanıcı çıkış yaptı');
        }
        
        session_destroy();
        header("Location: /public/login.php");
        exit();
    }

    public static function isLoggedIn() {
        return isset($_SESSION['user_id']) && self::startSecureSession();
    }

    public static function requireLogin() {
        if (!self::isLoggedIn()) {
            header("Location: /public/login.php");
            exit();
        }
    }

    public static function hasRole($required_role) {
        if (!self::isLoggedIn()) return false;
        
        $user_role = $_SESSION['role'];
        
        // Owner has all permissions
        if ($user_role === self::ROLE_OWNER) return true;
        
        return $user_role === $required_role;
    }

    public static function requireRole($required_role) {
        if (!self::hasRole($required_role)) {
            http_response_code(403);
            die("Bu işlem için yetkiniz bulunmamaktadır.");
        }
    }

    public static function canAccessBranch($branch_id) {
        if (!self::isLoggedIn()) return false;
        
        // Owner can access all branches
        if ($_SESSION['role'] === self::ROLE_OWNER) return true;
        
        // Others can only access their own branch
        return $_SESSION['branch_id'] == $branch_id;
    }

    public static function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    public static function validateCSRFToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }

    public static function hashPassword($password) {
        // ARGON2ID if available, otherwise use BCRYPT
        if (defined('PASSWORD_ARGON2ID')) {
            return password_hash($password, PASSWORD_ARGON2ID);
        } else {
            return password_hash($password, PASSWORD_BCRYPT);
        }
    }

    private static function getUserIP() {
        if (isset($_SERVER['HTTP_CF_CONNECTING_IP'])) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } elseif (isset($_SERVER['HTTP_X_REAL_IP'])) {
            return $_SERVER['HTTP_X_REAL_IP'];
        } else {
            return $_SERVER['REMOTE_ADDR'];
        }
    }

    private static function logFailedAttempt($ip) {
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO login_attempts (ip_address, failed_attempts, last_attempt) 
                                VALUES (?, 1, NOW()) 
                                ON DUPLICATE KEY UPDATE 
                                failed_attempts = failed_attempts + 1, last_attempt = NOW()");
            $stmt->execute([$ip]);
        } catch (Exception $e) {
            error_log("Failed to log login attempt: " . $e->getMessage());
        }
    }

    private static function clearFailedAttempts($ip) {
        try {
            $db = getDB();
            $stmt = $db->prepare("DELETE FROM login_attempts WHERE ip_address = ?");
            $stmt->execute([$ip]);
        } catch (Exception $e) {
            error_log("Failed to clear login attempts: " . $e->getMessage());
        }
    }

    public static function logActivity($user_id, $action_type, $target_table, $record_id, $description = '') {
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO activity_log 
                                (user_id, action_type, target_table, record_id, description, ip_address, created_at) 
                                VALUES (?, ?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$user_id, $action_type, $target_table, $record_id, $description, self::getUserIP()]);
        } catch (Exception $e) {
            error_log("Failed to log activity: " . $e->getMessage());
        }
    }
}

// Auto-start secure session
Auth::startSecureSession();
?>